import 'package:flutter/gestures.dart';
import 'package:flutter/material.dart';
import 'package:flutter_widget_from_html/flutter_widget_from_html.dart';
import 'package:get/get.dart';
import 'package:viserpay/core/helper/string_format_helper.dart';
import 'package:viserpay/core/route/route.dart';
import 'package:viserpay/core/utils/dimensions.dart';
import 'package:viserpay/core/utils/my_color.dart';
import 'package:viserpay/core/utils/my_strings.dart';
import 'package:viserpay/core/utils/style.dart';
import 'package:viserpay/data/controller/nominee/nominee_controller.dart';
import 'package:viserpay/data/controller/savings/savings_controller.dart';
import 'package:viserpay/data/repo/nominee/nominee_repo.dart';
import 'package:viserpay/data/repo/saving/saving_repo.dart';
import 'package:viserpay/data/services/api_service.dart';
import 'package:viserpay/view/components/app-bar/custom_appbar.dart';
import 'package:viserpay/view/components/buttons/gradient_rounded_button.dart';
import 'package:viserpay/view/components/dropdown/generic_drop_down.dart';
import 'package:skeletonizer/skeletonizer.dart';
import 'package:viserpay/data/model/savings/savings_response_model.dart' as savings;
import 'package:viserpay/view/components/global/history_icon_widget.dart';
import 'package:viserpay/view/components/snack_bar/show_custom_snackbar.dart';

class SavingsHomeScreen extends StatefulWidget {
  const SavingsHomeScreen({super.key});

  @override
  State<SavingsHomeScreen> createState() => _SavingsHomeScreenState();
}

class _SavingsHomeScreenState extends State<SavingsHomeScreen> {
  bool isTermsAndConditionsChecked = false;
  @override
  void initState() {
    Get.put(ApiClient(sharedPreferences: Get.find()));
    Get.put(SavingRepo(apiClient: Get.find()));
    Get.put(NomineeRepo(apiClient: Get.find()));
    Get.put(NomineeController(nomineeRepo: Get.find()));
    final controller = Get.put(SavingsController(savingRepo: Get.find(), nomineeController: Get.find()));
    super.initState();
    WidgetsBinding.instance.addPostFrameCallback((timeStamp) {
      controller.initializeData();
    });
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      backgroundColor: MyColor.colorWhite,
      appBar: CustomAppBar(
        title: MyStrings.savings,
        isTitleCenter: true,
        elevation: 0.1,
        action: [
          HistoryWidget(routeName: RouteHelper.savingHistoryScreen),
          const SizedBox(width: Dimensions.space20),
        ],
      ),
      body: GetBuilder<SavingsController>(
        builder: (controller) {
          return Skeletonizer(
            enabled: controller.isLoading,
            containersColor: MyColor.colorGrey2,
            effect: ShimmerEffect(baseColor: MyColor.colorGrey2, highlightColor: MyColor.borderColor),
            child: SingleChildScrollView(
              physics: const BouncingScrollPhysics(parent: AlwaysScrollableScrollPhysics()),
              padding: Dimensions.defaultPaddingHV,
              child: Column(
                crossAxisAlignment: CrossAxisAlignment.start,
                children: [
                  GenericDropdown<savings.Interval>(
                    title: MyStrings.selectTenure,
                    list: controller.tenureList,
                    onChanged: (value) {
                      if (value != null) {
                        controller.selectTenure(value);
                      }
                    },
                    displayItem: (item) => item.name ?? "",
                    selectedValue: controller.selectedTenure,
                  ),
                  const SizedBox(height: Dimensions.space15),
                  GenericDropdown<savings.Interval>(
                    title: MyStrings.selectInterval,
                    list: controller.intervalList,
                    onChanged: (value) {
                      if (value != null) {
                        controller.selectInterval(value);
                      }
                    },
                    displayItem: (item) => item.name ?? "",
                    selectedValue: controller.selectedInterval,
                  ),
                  const SizedBox(height: Dimensions.space15),
                  GenericDropdown<String>(
                    title: MyStrings.selectInstallmentAmount,
                    list: controller.installmentAmountList,
                    onChanged: (value) {
                      if (value != null) {
                        controller.selectInstallmentAmount(value);
                      }
                    },
                    displayItem: (item) => item == MyStrings.selectOne ? MyStrings.selectOne : "${controller.currencySym}${StringConverter.formatNumber(item, precision: 1)}",
                    selectedValue: controller.selectedInstallmentAmount,
                  ),
                  const SizedBox(height: Dimensions.space15),
                  Row(
                    children: [
                      Checkbox(
                        value: isTermsAndConditionsChecked,
                        onChanged: (value) {
                          setState(() {
                            isTermsAndConditionsChecked = value ?? false;
                          });
                        },
                        activeColor: MyColor.primaryColor,
                        side: BorderSide(color: MyColor.primaryColor),
                        shape: RoundedRectangleBorder(borderRadius: BorderRadius.circular(Dimensions.defaultRadius)),
                      ),
                      Expanded(
                        child: Text.rich(
                          TextSpan(children: [
                            TextSpan(
                              text: "${MyStrings.savingAgreementTerm.tr} ",
                              style: lightDefault.copyWith(),
                              recognizer: TapGestureRecognizer()
                                ..onTap = () {
                                  setState(() {
                                    isTermsAndConditionsChecked = true;
                                  });
                                },
                            ),
                            TextSpan(
                              text: MyStrings.termsAndConditions.tr,
                              style: semiBoldDefault.copyWith(color: MyColor.primaryColor),
                              recognizer: TapGestureRecognizer()..onTap = () => Get.toNamed(RouteHelper.savingsTermsConditionScreen),
                            ),
                          ]),
                        ),
                      ),
                    ],
                  ),
                  const SizedBox(height: Dimensions.space40),
                  Skeleton.replace(
                    replacement: Container(height: 60, width: double.infinity, decoration: BoxDecoration(color: MyColor.colorGrey2, borderRadius: BorderRadius.circular(Dimensions.largeRadius))),
                    child: GradientRoundedButton(
                      text: MyStrings.continue_,
                      press: () {
                        if (controller.selectedInterval?.id == "-1") {
                          CustomSnackBar.error(errorList: [MyStrings.selectSavingsInterval]);
                          return;
                        }
                        if (controller.selectedTenure?.id == "-1") {
                          CustomSnackBar.error(errorList: [MyStrings.selectSavingsTenure]);
                          return;
                        }
                        if (controller.selectedInstallmentAmount == MyStrings.selectOne) {
                          CustomSnackBar.error(errorList: [MyStrings.selectSavingsAmount]);
                          return;
                        }
                        if (isTermsAndConditionsChecked) {
                          controller.getOrganizationList();
                        } else {
                          CustomSnackBar.error(errorList: [MyStrings.readSavingTerm]);
                        }
                      },
                      isLoading: controller.isOrganizationListLoading,
                    ),
                  ),
                ],
              ),
            ),
          );
        },
      ),
    );
  }
}
